import numpy as np
import matplotlib.pyplot as plt


# Create the key locations (x, y per key) - similar to the Android keyboard:
def create_key_locations(key_w=1, key_h=1.5):
    key_locations = []
    # first row locations:
    for i in range(10):
        x = i*key_w
        y = 0
        key_locations.append([x,y])
    # second row locations:
    for i in range(9):
        x = i*key_w + key_w*0.5
        y = key_h
        key_locations.append([x,y])
    # third row locations:
    for i in range(7):
        x = i*key_w + key_w*1.5
        y = 2*key_h
        key_locations.append([x,y])   
    return np.array(key_locations)

# Visualises a keyboard layout, given the characters in order and the corresponding key locations:
def plot_keyboard(layout, key_locations, title='Keyboard'):
    max_x = np.max(key_locations[:,0])
    max_y = np.max(key_locations[:,1])

    plt.figure(figsize=((max_x, max_y)))
    ax = plt.subplot(111)
    ax.set_aspect('equal')
    plt.xlim((0, max_x))
    plt.ylim((0, max_y))
    ax.invert_yaxis()
    plt.axis('off')
    plt.scatter(key_locations[:,0], key_locations[:,1], c='k', s=0)
    for i, character in enumerate(layout):
        ax.annotate(character, key_locations[i], horizontalalignment='center', verticalalignment='center', fontsize=22)    
    plt.title(title+'\n', fontsize=22)
    #plt.tight_layout()
    plt.show()

# Measures the euclidean distance between two key locations:
def distance(key_location, key_location2):
    x,y = key_location
    x2,y2 = key_location2
    return np.sqrt((x-x2)**2 + (y-y2)**2)


# Computes the typing speed in words per minute (WPM), given the mean time interval between two key presses:
def wpm(mean_inter_key_time):
    return (1.0/mean_inter_key_time*60)/5